/*********************************************************************
* 
*   jLib2 :: cCircBuf_var_el.cpp
* 
*   Circular buffer. Takes variable length elements.  
*   Threadsafe.
*
*   Maybe this could be made slightly faster by 
*   padding the lengths of elements to multiples of 
*   4 (or higher powers of 2?) for memcpy.
* 
*	Every element is stored as a single continuous block.
*	In other words, the last element in the buffer is not
*	split into two pieces.  This wastes a little space but
*	simplifies things.
*
*   pHead: pointer to newest entry
*   pTail: pointer to oldest entry
*   buffer is empty when pHead and PTail are 0
*
*   2020 dec 19: expanded these notes
*	2020 dec 16: added put_without_write_and_get_buffer()
*   2020 dec 16: added pop(), qty()
*	2020 dec 15: begun
*
*	Copyright 2020 Robert Sacks.  https://mojoware.org
*
*********************************************************************/

#include "stdafx.h"
#include "cCircBuf_var_el.j.h"
#include <assert.h>
#include "cCriticalSectionRaii.j.h"

//-------------------------------------------------------------------------------------------------------
//   QTY
//-------------------------------------------------------------------------------------------------------
size_t cCircBuf_var_el :: qty ()
{
	cCriticalSectionRaii ( this->_cs );

	size_t uCount = 0;

	for ( sEntry * p = _pTail; p; p = p->pNext )
		uCount++;

	return uCount;
}


//-------------------------------------------------------------------------------------------------------
//   POP
//   discards value
//-------------------------------------------------------------------------------------------------------
bool cCircBuf_var_el :: pop ()
{
	cCriticalSectionRaii ( this->_cs );

	if ( _pTail )
	{
		_pTail = _pTail->pNext;

		if ( 0 == _pTail )
			_pHead = _pTail;

		return true;
	}

	else
		return false;
}


//-------------------------------------------------------------------------------------------------------
//   POP
//   return false and 0 < *pQtyWritten: element available but buffer needs to be larger
//   return false and 0 == *pQtyWritten: buffer empty
//   return true: element was returned
//-------------------------------------------------------------------------------------------------------
bool cCircBuf_var_el :: pop ( size_t * pQtyWritten, BYTE * pRet, size_t  uBufSize )
{
	cCriticalSectionRaii ( this->_cs );

	if ( 0 == _pTail )
	{
		*pQtyWritten = 0;
		return false;
	}

	else if ( uBufSize < _pTail->uPayloadLen )
	{
		*pQtyWritten = _pTail->uPayloadLen;
		return false;
	}

	else
	{
		// THIS MEMCPY IS THE ONLY EXPENSIVE 
		// OPERATION IN THIS CLASS WHEN READING:

		memcpy ( pRet, _pTail->aPayload, _pTail->uPayloadLen );
		*pQtyWritten = _pTail->uPayloadLen;
		_pTail = _pTail->pNext;

		if ( 0 == _pTail )
			_pHead = _pTail;

		return true;
	}
}


//-------------------------------------------------------------------------------------------------------
//   PUT WITHOUT WRITE AND GET BUFFER
//   Caller must lock and unlock!
//-------------------------------------------------------------------------------------------------------
bool cCircBuf_var_el :: put_without_write_and_get_buffer ( BYTE ** ppRet, /* BYTE * pA, */ size_t uLenA )
{
	// Caller must lock and unlock!

	if ( _uBufSize < uLenA )
	{
		assert(0);
		return false;
	}

	sEntry * pNew = 0;

	if ( 0 == _pHead )
	{
		// empty
		assert ( 0 == _pTail );
		pNew = (sEntry*) _pBuf;
		pNew->pNext = 0;
		_pHead = _pTail = pNew;
	}

	else
	{
		BYTE * pbEoBuf = _pBuf + _uBufSize;
		BYTE * pbEoOldHead = (BYTE*)_pHead + _pHead->size();
		size_t uSizeNew = sizeof(sEntry) + uLenA - 4;

		size_t uSizeRemaining = (size_t) ( pbEoBuf - pbEoOldHead ); uSizeRemaining;

		if ( uSizeNew <= (size_t) ( pbEoBuf - pbEoOldHead ) )
		{
			pNew = ( sEntry*) pbEoOldHead;
			_pHead->pNext = pNew;
			_pHead = pNew;
		}

		else
		{
			pNew = ( sEntry*) _pBuf;
			_pHead->pNext = pNew;
			_pHead = pNew;
		}

		// adjust tail if it is getting overwritten
		BYTE * pbEoNewHead = (BYTE*) pNew + uSizeNew;
		while ( pNew < _pTail && (BYTE*) _pTail <= pbEoNewHead )
			_pTail = _pTail->pNext;
	}

	// initialize new entry
	pNew->pNext = 0;
	pNew->uPayloadLen = uLenA;
	*ppRet = pNew->aPayload;

	return true;
}


//-------------------------------------------------------------------------------------------------------
//   PUT
//-------------------------------------------------------------------------------------------------------
bool cCircBuf_var_el :: put ( BYTE * pA, size_t uLenA )
{
	cCriticalSectionRaii ( this->_cs );

	BYTE * pDest;

	if ( put_without_write_and_get_buffer ( &pDest, uLenA ) )
	{
		// THIS MEMCPY IS THE ONLY EXPENSIVE 
		// OPERATION IN THIS CLASS WHEN WRITING:

		memcpy ( pDest, pA, uLenA );
		return true;
	}

	else
		return false;
}

