/***********************************************************************************************************************
/*
/*	cCrypt_rotate.j.cpp
/*
/*  For obfuscation only of blobs, strings, and integers.
/*	Super fast. Uses a combination of rotation, XOR, and 
/*	multiplication.  Note that the blob/string functions 
/*	process only integral number of 8-byte blocks and may
/*	leave as many as 7 bytes of plaintext at the end unless
/*	you pad input to make its length a multiple of 8 bytes.
/*
/*	Sample usage for a string.
/*
/*		// Pad input string to make it multiple of 8 bytes,
/*		// otherwise as many as 7 final bytes may be left as
/*		// plaintext.  If you're too lazy to count, like me, 
/*		// add 7 NULLs.
/*
/*		BYTE a [] = "My obfuscated message.\0\0\0\0\0\0\0";
/*
/*		cCrypt_rotate::encrypt ( a, ARRAYSIZE(a) );
/*
/*		cCrypt_rotate::decrypt ( a, ARRAYSIZE(a) );
/*
/*	For added security (not that this is terribly secure),
/*	replace the three default constants with your own values
/*  in the function calls.
/*
/*	This class also contains functions (not shown above) 
/*	to obfuscate integers.
/*
/*	2021 jun 03: non-template functions moved to cCrypt_rotate.j.cpp
/*  2021 may 02: some functions renamed, made static, made public
/*  2021 may 02: added parameters for uRot, uMult, and uXor
/*  2020 dec 23: begun
/*
/*	Copyright 2020-21 Robert Sacks.  https://mojoware.org
/*
/**********************************************************************************************************************/

#include "cCrypt_rotate.j.h"

//======================================================================================================================
// PROTOTYPES
//======================================================================================================================

//======================================================================================================================
// CODE
//======================================================================================================================

//----------------------------------------------------------------------------------------------------------------------
//  ENCRYPT -- BUFFER
//----------------------------------------------------------------------------------------------------------------------
void cCrypt_rotate :: encrypt ( void * pa, size_t size_in_bytes, 
	unsigned uRot, uint64_t uXor, uint64_t uMult )
{
	auto * p = ( uint64_t * ) pa;

	for ( size_t u = 0; 
		u < size_in_bytes / sizeof(*p);
		u ++ )
	{
		p[u] = encrypt_int ( p[u] + u * uMult, uRot, uXor );
	}
}


//----------------------------------------------------------------------------------------------------------------------
//  DECRYPT -- BUFFER
//----------------------------------------------------------------------------------------------------------------------
void cCrypt_rotate :: decrypt ( void * pa, size_t size_in_bytes, 
	unsigned uRot, uint64_t uXor, uint64_t uMult )
{
	uint64_t * p = ( uint64_t * ) pa;

	for ( size_t u = 0; 
		u < size_in_bytes / sizeof(uint64_t);
		u ++ )
	{
		p[u] = decrypt_int ( p[u], uRot, uXor ) - u * uMult;
	}
}